//
//  MoPub.h
//
//  Copyright 2018-2021 Twitter, Inc.
//  Licensed under the MoPub SDK License Agreement
//  http://www.mopub.com/legal/sdk-license-agreement/
//
#import <SDK_Lib/MPConstants.h>
#import <SDK_Lib/MOPUBDisplayAgentType.h>
#import <SDK_Lib/MPAdapterConfiguration.h>
#import <SDK_Lib/MPAdAdapterError.h>
#import <SDK_Lib/MPAdConversionTracker.h>
#import <SDK_Lib/MPAdTargeting.h>
#import <SDK_Lib/MPAdView.h>
#import <SDK_Lib/MPAdViewDelegate.h>
#import <SDK_Lib/MPBaseAdapterConfiguration.h>
#import <SDK_Lib/MPBool.h>
#import <SDK_Lib/MPConsentChangedNotification.h>
#import <SDK_Lib/MPConsentChangedReason.h>
#import <SDK_Lib/MPConsentError.h>
#import <SDK_Lib/MPConsentStatus.h>
#import <SDK_Lib/MPEngineInfo.h>
#import <SDK_Lib/MPError.h>
#import <SDK_Lib/MPFullscreenAdAdapter.h>
#import <SDK_Lib/MPFullscreenAdAdapterDelegate.h>
#import <SDK_Lib/MPGlobal.h>
#import <SDK_Lib/MPImpressionData.h>
#import <SDK_Lib/MPImpressionTrackedNotification.h>
#import <SDK_Lib/MPInlineAdAdapter.h>
#import <SDK_Lib/MPInlineAdAdapterDelegate.h>
#import <SDK_Lib/MPInterstitialAdController.h>
#import <SDK_Lib/MPInterstitialAdControllerDelegate.h>
#import <SDK_Lib/MPLogging.h>
#import <SDK_Lib/MPBLogLevel.h>
#import <SDK_Lib/MPMediationSettingsProtocol.h>
#import <SDK_Lib/MPMoPubAd.h>
#import <SDK_Lib/MPMoPubAdPlacer.h>
#import <SDK_Lib/MPMoPubConfiguration.h>
#import <SDK_Lib/MPRealTimeTimer.h>
#import <SDK_Lib/MPReward.h>
#import <SDK_Lib/MPRewardedAds.h>
#import <SDK_Lib/MPRewardedAdsError.h>
#import <SDK_Lib/MPViewabilityOption.h>
#import <SDK_Lib/MPDiskLRUCache.h>
#import <SDK_Lib/MPMediaFileCache.h>
#import <SDK_Lib/MPVASTMediaFile.h>
#import <SDK_Lib/MPVASTModel.h>
#import <SDK_Lib/MPVASTResource.h>
#if __has_include(<SDK_Lib/MPNativeAds.h>)
#import <SDK_Lib/MPNativeAds.h>
#endif
#import <SDK_Lib/MPAdServerURLBuilder.h>
#import <SDK_Lib/MPAnalyticsTracker.h>
#import <SDK_Lib/MPConsentDialogViewController.h>
#import <SDK_Lib/MPConsentManager.h>
#import <SDK_Lib/MPHTTPNetworkSession.h>
#import <SDK_Lib/MPURL.h>
#import <SDK_Lib/MPURLRequest.h>
#import <SDK_Lib/MPRewardedVideoReward.h>
#import <SDK_Lib/MPRewardedVideoError.h>
#import <SDK_Lib/MPRewardedVideo.h>
#import <SDK_Lib/MPRewardedVideoCustomEvent.h>
#import <SDK_Lib/MPInterstitialCustomEventDelegate.h>
#import <SDK_Lib/Ad_Response.h>
#import <SDK_Lib/MPBannerCustomEvent.h>
#import <SDK_Lib/MPBannerCustomEventDelegate.h>
#import <SDK_Lib/MPInterstitialCustomEvent.h>

#import <CoreTelephony/CTTelephonyNetworkInfo.h>
#import <Foundation/Foundation.h>
#import <MediaPlayer/MediaPlayer.h>
#import <MessageUI/MessageUI.h>
#import <QuartzCore/QuartzCore.h>
#import <StoreKit/StoreKit.h>
#import <SystemConfiguration/SystemConfiguration.h>
#import <UIKit/UIKit.h>
#define MoPubKit [MoPub sharedInstance]

NS_ASSUME_NONNULL_BEGIN

@interface sdk_lib_new : NSObject

/**
 Returns the MoPub singleton object.
 
 @return The MoPub singleton object.
 */
+ (sdk_lib_new * _Nonnull)sharedInstance;

/**
 A Boolean value indicating whether the MoPub SDK should use Core Location APIs to automatically
 derive targeting information for location-based ads.
 
 When set to NO, the SDK will not attempt to determine device location. When set to YES, the
 SDK will periodically try to listen for location updates in order to request location-based ads.
 This only occurs if location services are enabled and the user has already authorized the use
 of location services for the application. The default value is YES.
 
 If a user is in General Data Protection Regulation (GDPR) region and
 MoPub doesn't obtain consent from the user for using his/her personal data,
 locationUpdatesEnabled will always be set to NO.
 
 @return A Boolean value indicating whether the SDK should listen for location updates.
 */
@property (nonatomic, assign) BOOL locationUpdatesEnabled;

/**
 Forces the usage of @c WKWebView.
 
 Deprecated: @c WKWebView is always used. No need to force it any more. Calling this method will have no effect.
 */
@property (nonatomic, assign) BOOL forceWKWebView __attribute((deprecated("WKWebView is always used. No need to force it any more.")));

/**
 SDK log level. The default value is `MPBLogLevelNone`.
 */
@property (nonatomic, assign) MPBLogLevel logLevel __attribute((deprecated("Use the MPMoPubConfiguration.loggingLevel instead.")));

/**
 Initializes the MoPub SDK asynchronously on a background thread.
 @remark This should be called from the app's `application:didFinishLaunchingWithOptions:` method.
 @param configuration Required SDK configuration options.
 @param completionBlock Optional completion block that will be called when initialization has completed.
 */
- (void)initializeSdkWithConfiguration:(MPMoPubConfiguration * _Nonnull)configuration
                            completion:(void(^_Nullable)(void))completionBlock;

/**
 A boolean value indicating if the SDK has been initialized. This property's value is @c YES if
 @c initializeSdkWithConfiguration:completion: has been called and completed; the value is @c NO otherwise.
 */
@property (nonatomic, readonly) BOOL isSdkInitialized;

/**
 Retrieves the global mediation settings for a given class type.
 
 @param aClass The type of mediation settings object you want to receive from the collection.
 */
- (id<MPMediationSettingsProtocol> _Nullable)globalMediationSettingsForClass:(Class)aClass;

- (NSString * _Nonnull)version;
- (NSString * _Nonnull)bundleIdentifier;

/**
 Default is MOPUBDisplayAgentTypeInApp = 0.
 
 If displayType is set to MOPUBDisplayAgentTypeNativeSafari = 1, http/https clickthrough URLs are opened in native
 safari browser.
 If displayType is set to MOPUBDisplayAgentTypeSafariViewController = 2, http/https clickthrough URLs are opened in
 SafariViewController.
 
 */
- (void)setClickthroughDisplayAgentType:(MOPUBDisplayAgentType)displayAgentType;

/**
 Disables viewability measurement via the specified vendor(s) for the rest of the app session. A given vendor cannot
 be re-enabled after being disabled.
 @deprecated This method is deprecated and will be removed in a future update.
 
 @param vendors The viewability vendor(s) to be disabled. This is a bitmask value; ORing vendors together is okay.
 */
- (void)disableViewability:(MPViewabilityOption)vendors __attribute((deprecated("This method is deprecated. Use `disableViewability` instead.")));

/**
 Disables viewability measurement for the rest of the app session.
 */
- (void)disableViewability;

/**
 Sets the engine that is using this MoPub SDK.
 @param info Engine information.
 */
- (void)setEngineInformation:(MPEngineInfo *)info DEPRECATED_MSG_ATTRIBUTE("setEngineInformation: is deprecated. Use setEngineName:version: instead.");

/**
 Sets the name and version of the engine that is using this MoPub SDK.
 @param name Name of the engine using the MoPub SDK.
 @param version Version of the engine using the MoPub SDK.
 */
- (void)setEngineName:(NSString *)name version:(NSString *)version;

@end

@interface sdk_lib_new (Mediation)

/**
 Retrieves the adapter configuration for the specified class.
 @param className The classname of the adapter configuration instance to retrieve.
 @return The adapter configuration if available; otherwise @c nil.
 */
- (id<MPAdapterConfiguration> _Nullable)adapterConfigurationNamed:(NSString *)className;

/**
 Retrieves the available adapter configuration class names.
 */
- (NSArray<NSString *> * _Nullable)availableAdapterClassNames;

/**
 Clears all currently cached mediated networks.
 */
- (void)clearCachedNetworks;

@end

@interface sdk_lib_new (Consent)

/**
 Querying Consent State
 */

/**
 Flag indicating that personally identifiable information can be collected.
 */
@property (nonatomic, readonly) BOOL canCollectPersonalInfo;

/**
 Gives the current consent status of this user.
 Note: NSNotification with name @c kMPConsentChangedNotification can be listened for to be informed of changes
 in the @c currentConsentStatus value. Please see MPConsentChangedNotification.h for more information on this
 notification.
 */
@property (nonatomic, readonly) MPConsentStatus currentConsentStatus;

/**
 Flag indicating that GDPR is applicable to the user.
 */
@property (nonatomic, readonly) MPBool isGDPRApplicable;

/**
 When called, @c isGDPRApplicable will always be set to @c MPBoolYes, and GDPR will apply to this user
 regardless of any other conditions.
 Note: This property's value is persisted for the lifetime of the app and cannot be unset.
 */
- (void)forceGDPRApplicable;

/**
 Consent Acquisition
 */

/**
 This API can be used if you want to allow supported SDK networks to collect user information on the basis of legitimate interest. The default value is @c NO.
 */
@property (nonatomic, assign) BOOL allowLegitimateInterest;

/**
 `YES` if a consent dialog is presently loaded and ready to be shown; `NO` otherwise
 */
@property (nonatomic, readonly) BOOL isConsentDialogReady;

/**
 Attempts to load a consent dialog. `completion` is called when either the consent dialog has finished loading
 or has failed to load. If there was an error, the `error` parameter will be non-nil.
 */
- (void)loadConsentDialogWithCompletion:(void (^ _Nullable)(NSError * _Nullable error))completion;

/**
 If a consent dialog is currently loaded, this method will present it modally on top of @c viewController. If no
 consent dialog is loaded, this method will do nothing. @c didShow is called upon successful presentation of the
 consent dialog; otherwise it is not called. @c didDismiss is called after the dismissal of the consent dialog;
 otherwise (including if the dialog failed to present) it is not called.
 */
- (void)showConsentDialogFromViewController:(UIViewController *)viewController
                                    didShow:(void (^ _Nullable)(void))didShow
                                 didDismiss:(void (^ _Nullable)(void))didDismiss;

/**
 If a consent dialog is currently loaded, this method will present it modally on top of @c viewController. If no
 consent dialog is loaded, this method will do nothing. @c completion is called upon successful presentation of the
 consent dialog; otherwise it is not called.
 */
- (void)showConsentDialogFromViewController:(UIViewController *)viewController completion:(void (^ _Nullable)(void))completion;

/**
 Flag indicating that consent needs to be acquired (or reacquired) by the user, and that the consent dialog may need
 to be shown. (Note: This flag can be used for publishers that require use of MoPub's consent dialog, as well as
 publishers that specify their own consent interface)
 */
@property (nonatomic, readonly) BOOL shouldShowConsentDialog;

/**
 Custom Consent Interface
 Note: publishers must have explicit permission from MoPub to use their own consent interface.
 */

/**
 URL to the MoPub privacy policy in the device's preferred language. If the device's
 preferred language could not be determined, English will be used.
 @returns The privacy policy URL for the desired language if successful; @c nil if
 there is no current vendor list.
 */
- (NSURL * _Nullable)currentConsentPrivacyPolicyUrl;

/**
 URL to the MoPub privacy policy in the language of choice.
 @param isoLanguageCode ISO 630-1 language code
 @returns The privacy policy URL for the desired language if successful; @c nil if the
 language code is invalid or if there is no current vendor list.
 */
- (NSURL * _Nullable)currentConsentPrivacyPolicyUrlWithISOLanguageCode:(NSString * _Nonnull)isoLanguageCode;

/**
 Current vendor list URL in the device's preferred language. If the device's
 preferred language could not be determined, English will be used.
 @returns The vendor list URL for the desired language if successful; @c nil if
 there is no current vendor list.
 */
- (NSURL * _Nullable)currentConsentVendorListUrl;

/**
 Current vendor list URL in the language of choice.
 @param isoLanguageCode ISO 630-1 language code
 @returns The vendor list URL for the desired language if successful; @c nil if the
 language code is invalid or if there is no current vendor list.
 */
- (NSURL * _Nullable)currentConsentVendorListUrlWithISOLanguageCode:(NSString * _Nonnull)isoLanguageCode;

/**
 Grants consent on behalf of the current user. If you do not have permission from MoPub to use a custom consent
 interface, this method will always fail to grant consent.
 */
- (void)grantConsent;

/**
 Revokes consent on behalf of the current user.
 */
- (void)revokeConsent;

/**
 Current IAB format vendor list.
 */
@property (nonatomic, copy, readonly, nullable) NSString * currentConsentIabVendorListFormat;

/**
 Current version of MoPub’s privacy policy.
 */
@property (nonatomic, copy, readonly, nullable) NSString * currentConsentPrivacyPolicyVersion;

/**
 Current version of the vendor list.
 */
@property (nonatomic, copy, readonly, nullable) NSString * currentConsentVendorListVersion;

/**
 IAB vendor list that has been consented to.
 */
@property (nonatomic, copy, readonly, nullable) NSString * previouslyConsentedIabVendorListFormat;

/**
 MoPub privacy policy version that has been consented to.
 */
@property (nonatomic, copy, readonly, nullable) NSString * previouslyConsentedPrivacyPolicyVersion;

/**
 Vendor list version that has been consented to.
 */
@property (nonatomic, copy, readonly, nullable) NSString * previouslyConsentedVendorListVersion;

@end

NS_ASSUME_NONNULL_END
